/*Lechner

  UDPtoIR - v1.0

  extended with Ethernet DHCP and optional 24AA025E48 EEPROM as source for the MAC address
  - patrik.mayer@codm.de, 2017

*/

//// ---------------- config
#define MAX_CHANNEL 16

#define udpPort  7000
//#define DEBUG
//#define UDPDEBUG
//#define USE_DMX_Serial
// --- arduino + ethernet config
// use the uuid of a 24AA025E48 eeprom attached to i2c
#define USE_EEPROM

// use dhcp: myip, mygw, mymask are ignored
#define USE_DHCP

// MAC address to use, overwritten if USE_EEPROM is defined
uint8_t mymac[6] = {  0x54, 0x55, 0x58, 0x10, 0x01, 0x39 };

//STATIC IP, ignored if USE_DHCP is defined
uint8_t myip[4] = { 192, 168, 123, 39 };
const uint8_t mygw[4] = { 192, 168, 123, 254 };// ip of gateway
const uint8_t mymask[4] = { 255, 255, 255, 0 };// subnet mask
//SPI
#define SS_SD_CARD   4
#define SS_ETHERNET 10

// --- ESP8266 config

// no need to configure anything as the ESP brings it's own MAC address,
// USE_EEPROM and mymac are meaningless and dhcp is always used

// static IP for ESP not implemented yet
// simply choose ESP8266 from the Tools Menu,

//// -----------------------




#ifdef ESP8266
#include <ESPDMX.h>
#include <ESP8266WiFi.h>
#include <WiFiUDP.h>

class EthernetUDP: public WiFiUDP { };

#define DMX_SIZE  128
int status = WL_IDLE_STATUS;
const char* ssid = "dd-wrt_57b";  //  your network SSID (name)
const char* pass = "secret";       // your network password
DMXESPSerial dmx;
int cnt2 = 0;

#else
#include <SPI.h>
#include <Ethernet.h>
#include <EthernetUdp.h>         // UDP library from: bjoern@cs.stanford.edu 12/30/2008

#ifdef USE_DMX_Serial
#include <DMXSerial.h>
#define dmx DMXSerial
#define DMX_SIZE  128     // also CHANGE DMXSERIAL_MAX in DMXSerial.h  to save memory
#else
#include <DmxSimple.h>           // use DmxSimple from the libraries folder
#define dmx DmxSimple
//#define DMX_SIZE  128  << CHANGE it in  DmxSimple.h !
#endif

#ifdef USE_EEPROM
#include <Wire.h>                // for MAC from 24AA02E48
#define UUID_I2C_ADDRESS 0x50    // i2c Address of UUID EEPROM
#endif


#endif


EthernetUDP Udp;

int cnt = 0;

int RECV_PIN = 8;
//int BUTTON_PIN = 12;
int STATUS_PIN = 13;

uint32_t lastTime = 0;
uint32_t lastTime2 = 0;
uint32_t lastTime3 = 0;
uint32_t ct = 0;
uint32_t dt = 0;
uint32_t maxdt = 0;
uint32_t dtsum = 0;
uint32_t dtcnt = 0;


// buffer for receiving data
#define maxPacketSize (UDP_TX_PACKET_MAX_SIZE*2)  //MAX Message Size
char packetBuffer[maxPacketSize]; //buffer to hold incoming packet,

volatile uint8_t dmxValues[DMX_SIZE];
volatile uint8_t dmxSpeeds[DMX_SIZE];

#ifdef UDPDEBUG
#define DEBUG_PRINTln(x) {Udp.beginPacket(myip, udpPort); Udp.println(x);  Udp.endPacket();}
#define DEBUG_PRINT(x)  {Udp.beginPacket(myip, udpPort); Udp.print(x);  Udp.endPacket();}
#define DEBUG_PRINT2(x, y) {Udp.beginPacket(myip, udpPort); Udp.print(x, y);  Udp.endPacket();}
#else
#ifdef DEBUG
#define DEBUG_PRINTln(x)  Serial.println (x)
#define DEBUG_PRINT(x)  Serial.print (x)
#define DEBUG_PRINT2(x, y)  Serial.print (x, y)
#else
#define DEBUG_PRINT(x)
#define DEBUG_PRINTln(x)
#define DEBUG_PRINT2(x, y)
#endif
#endif

void _dmxInit(int maxChannels) {
#ifdef ESP8266
  dmx.init(maxChannels);
#else
#ifdef USE_DMX_Serial
  dmx.init(DMXController);
  dmx.maxChannel(maxChannels);
#else

  /* The most common pin for DMX output is pin 3, which DmxSimple
   ** uses by default. If you need to change that, do it here. */
  dmx.usePin(3);
  /* DMX devices typically need to receive a complete set of channels
  ** even if you only need to adjust the first channel. You can
  ** easily change the number of channels sent here. If you don't
  ** do this, DmxSimple will set the maximum channel number to the
  ** highest channel you dmx.write() to. */
  dmx.maxChannel(maxChannels);
#endif
#endif
}

uint8_t _dmxRead(int channel) {
#if (defined(USE_DMX_Serial) || defined(ESP8266))
  return dmx.read(channel);
#else
  return dmx.getValue(channel);
#endif
}

void _dmxWrite (int channel, uint8_t value) {
  dmx.write(channel, value);
}


// ----------------- Arduino SETUP -------------------------
void setup() {
#ifdef DEBUG
#if defined(__AVR_ATmega32U4__)
  while (not(Serial)) {};
#endif
#ifndef USE_DMX_Serial
  Serial.begin(115200);
#endif
#endif
#ifndef UDPDEBUG
  DEBUG_PRINTln(F("Starting up..."));
#endif

  //IR
  //pinMode(BUTTON_PIN, INPUT);
  pinMode(STATUS_PIN, OUTPUT);

#ifdef ESP8266
  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, pass);

  // Wait for connect to AP
  DEBUG_PRINT(F("[Connecting]"));
  DEBUG_PRINT(ssid);
  int tries = 0;
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    DEBUG_PRINT(".");
    tries++;
    if (tries > 30) {
      break;
    }
  }
  /* print the DHCP IP as Serial is initialized  */
  DEBUG_PRINT(F("My IP address: "));
  DEBUG_PRINTln(WiFi.localIP());
#else

#ifdef USE_EEPROM
  //read bytes for MAC from 24AA025E48 via i2c
  Wire.begin();
  mymac[0] = readUUIDRegister(0xFA);
  mymac[1] = readUUIDRegister(0xFB);
  mymac[2] = readUUIDRegister(0xFC);
  mymac[3] = readUUIDRegister(0xFD);
  mymac[4] = readUUIDRegister(0xFE);
  mymac[5] = readUUIDRegister(0xFF);

  char tmpBuf[17];
  sprintf(tmpBuf, "%02X:%02X:%02X:%02X:%02X:%02X", mymac[0], mymac[1], mymac[2], mymac[3], mymac[4], mymac[5]);
  DEBUG_PRINT(F("My MAC: "));
  DEBUG_PRINTln(tmpBuf);
#endif //USE_EEPROM

  pinMode(SS_SD_CARD, OUTPUT);
  pinMode(SS_ETHERNET, OUTPUT);
  digitalWrite(SS_SD_CARD, HIGH);  // SD Card not active
  digitalWrite(SS_ETHERNET, LOW); // Ethernet active

#ifdef USE_DHCP
#ifdef SET_HOSTNAME_AVAILABLE
  const char * hostName = "eth2dmx";
  Ethernet.setHostName(hostName);
#endif
  DEBUG_PRINT(F("Set hostname to: "));
  DEBUG_PRINTln(Ethernet.getHostName());
  
  while (Ethernet.begin(mymac) == 0) {
    DEBUG_PRINTln(F("Failed to configure Ethernet using DHCP"));
    DEBUG_PRINTln(F("Retry..."));
  }
#else
  Ethernet.begin(mymac, myip, mygw, mymask);
#endif

  // print the Ethernet IP as Serial is initialized anyway
#endif //ESP8266

  Udp.begin(udpPort);

  delay(2000);
#ifdef UDPDEBUG
  myip[3] = 137;
#endif
#ifndef ESP8266
  DEBUG_PRINT(F("My IP address: "));
  DEBUG_PRINTln(Ethernet.localIP());
#endif
  _dmxInit(MAX_CHANNEL);
}


void dmxValueWrite(int channel, uint8_t value) {
  if ((channel > 0) && (channel <= DMX_SIZE)) {
    if (value < 0) value = 0;
    if (value > 255) value = 255;
    dmxValues[channel - 1] = value;
  }
}

bool isOnOffSpeed(int channel) {
  int oldValue = _dmxRead(channel);
  bool res = (abs(int(dmxValues[channel - 1]) - oldValue) > 20);
  return res;
}


void dmxSpeedsWrite(int channel, uint8_t value) {
  if ((channel > 0) && (channel <= DMX_SIZE)) {
    if (value < 0) value = 0;
    if (value > 255) value = 255;
    dmxSpeeds[channel - 1] = value;
  }
}

uint8_t currStep = 0;
#ifdef DEBUG
uint32_t NULLTIME = 0;
#endif
int32_t lastnv = 0;

void stepDMX() {
  uint8_t oldValue, chStep, chInc;
  int32_t nv;
  for (int ch = 1; ch <= DMX_SIZE; ch++) {
#ifdef DEBUG
    if (ch == 1) {
      oldValue = _dmxRead(ch);
      if (oldValue == 0) {
        NULLTIME = millis();
      }
    }
#endif
    chStep = dmxSpeeds[ch - 1] % 100;
    chInc = dmxSpeeds[ch - 1] / 100;
    if (chInc == 0) {
      chInc = 1;
    }  else if (chInc == 1) {
      chInc = 4;
    }  else if (chInc == 2) {
      chInc = 8;
      if (chStep == 55) {
        chInc = 255;
      }
    }
    if ((chStep == 0)  || ((currStep % chStep) == 0)) {
      oldValue = _dmxRead(ch);
      bool ok = false;
      if (dmxValues[ch - 1] > oldValue) {
        nv = int32_t(oldValue) + chInc;
        if (nv > dmxValues[ch - 1]) {
          nv = dmxValues[ch - 1];
        }
        _dmxWrite(ch, nv);
        ok = true;
      } else if (dmxValues[ch - 1] < oldValue) {
        nv = int32_t(oldValue) - chInc;
        if (nv < dmxValues[ch - 1]) {
          nv = dmxValues[ch - 1];
        }
        _dmxWrite(ch, nv);
        ok = true;
      }

      if (ok && (ch == 1) && (lastnv != nv)) {
        lastnv = nv;
      }

    }
  }
};


uint8_t toByte(char inp)
{
  uint8_t result = 0;
  if (isdigit(inp))
    result = inp - '0';
  else if (isupper(inp))
    result = inp - 'A' + 10;
  else if (islower(inp))
    result = inp - 'a' + 10;
  return result;
}


uint32_t timeDiff(uint32_t from, uint32_t to) {
  uint32_t res;
  if (from <= to) {
    res = to - from;
  } else {
    res = to + (0xFFFFFFFF - from);
  }
  return res;
}

uint8_t c255(uint8_t proz, uint8_t gamma) {
  uint32_t p2 = proz;
  if (gamma == 3) {
    return p2 * p2 * p2 * 255 / 1000000;
  } else if (gamma == 2) {
    return p2 * p2 * 255 / 10000;
  } else if (gamma == 1) {
    uint32_t tmp = p2 * p2 * 255 / 10000;
    tmp = tmp + (p2 * 255 / 100);
    return tmp / 2;
  } else {
    return p2 * 255 / 100;
  }
};

void ParseUDP() {
  int packetSize = Udp.parsePacket();
  if (packetSize)
  {
    // read the packet into packetBufffer
    Udp.read(packetBuffer, maxPacketSize);
    if (packetSize > maxPacketSize) {
      packetSize = maxPacketSize;
    }
    DEBUG_PRINT(F("UDP Rcv: "));
    DEBUG_PRINTln(packetBuffer);


    char conv[10];
    int p1 = 0;
    int p2 = 0;
    int startChannel = 0;
    long newValue = 0;
    int updSp = 0;
    int onoffSpeed = 0;
    int gamma = 0;

    if ((packetSize > 3) &&
        (packetBuffer[0] == 'D')  &&
        (packetBuffer[0 + 1] == 'M')  &&
        (packetBuffer[0 + 2] == 'X')  &&
        ((packetBuffer[0 + 3] == 'C') ||
         (packetBuffer[0 + 3] == 'R') ||
         (packetBuffer[0 + 3] == 'P') ||
         (packetBuffer[0 + 3] == 'V') ||
         (packetBuffer[0 + 3] == 'W'))) {
      DEBUG_PRINTln(F("UDP DATA OK2"));
      DEBUG_PRINTln(packetSize);

      p1 = p1 + 4;
      conv[0] = '\0';
      while ((p1 < packetSize) && (packetBuffer[p1] >= '0') && (packetBuffer[p1] <= '9')) {
        conv[p2++] = packetBuffer[p1];
        p1++;
      }
      conv[p2] = '\0';
      startChannel = atoi(conv);

      p1++;
      p2 = 0;
      conv[0] = '\0';
      while ((p1 < packetSize) && (packetBuffer[p1] >= '0') && (packetBuffer[p1] <= '9')) {
        conv[p2++] = packetBuffer[p1];
        p1++;
      }
      conv[p2] = '\0';
      newValue = atol(conv);

      p1++;
      p2 = 0;
      conv[0] = '\0';
      while ((p1 < packetSize) && (packetBuffer[p1] >= '0') && (packetBuffer[p1] <= '9')) {
        conv[p2++] = packetBuffer[p1];
        p1++;
      }
      conv[p2] = '\0';
      updSp = atoi(conv);
      if (updSp == 0) {
        updSp = 2;
      }

      p1++;
      p2 = 0;
      conv[0] = '\0';
      while ((p1 < packetSize) && (packetBuffer[p1] >= '0') && (packetBuffer[p1] <= '9')) {
        conv[p2++] = packetBuffer[p1];
        p1++;
      }
      conv[p2] = '\0';
      gamma = atoi(conv);

      p1++;
      p2 = 0;
      conv[0] = '\0';
      while ((p1 < packetSize) && (packetBuffer[p1] >= '0') && (packetBuffer[p1] <= '9')) {
        conv[p2++] = packetBuffer[p1];
        p1++;
      }
      conv[p2] = '\0';
      onoffSpeed = atoi(conv);
      if (onoffSpeed == 0) {
        onoffSpeed = updSp;
      }

      DEBUG_PRINT(F("Channel: "));
      DEBUG_PRINTln(startChannel);
      DEBUG_PRINT(F("newValue: "));
      DEBUG_PRINTln(newValue);
      DEBUG_PRINT(F("Speed: "));
      DEBUG_PRINTln(updSp);
      DEBUG_PRINT(F("onoff Speed: "));
      DEBUG_PRINTln(onoffSpeed);
      DEBUG_PRINT(F("old Value: "));
      DEBUG_PRINTln(_dmxRead(startChannel));
      DEBUG_PRINT(packetBuffer[0 + 3]);
      DEBUG_PRINT(" ");
      if  (packetBuffer[0 + 3] == 'R') {
        dmxValueWrite(startChannel, c255(newValue % 1000, gamma));
        newValue = newValue / 1000;
        dmxValueWrite(startChannel + 1, c255(newValue % 1000, gamma));
        newValue = newValue / 1000;
        dmxValueWrite(startChannel + 2, c255(newValue, gamma));
        if (isOnOffSpeed(startChannel) || isOnOffSpeed(startChannel + 1)  || isOnOffSpeed(startChannel + 2)) {
          updSp = onoffSpeed;
        }
        dmxSpeedsWrite(startChannel, updSp);
        dmxSpeedsWrite(startChannel + 1, updSp);
        dmxSpeedsWrite(startChannel + 2, updSp);
      } else if  (packetBuffer[0 + 3] == 'P') {
        DEBUG_PRINTln(c255(newValue, gamma));
        dmxValueWrite(startChannel, c255(newValue, gamma));
        if (isOnOffSpeed(startChannel)) {
          updSp = onoffSpeed;
        }
        dmxSpeedsWrite(startChannel, updSp);
      } else if  (packetBuffer[0 + 3] == 'W') {
        dmxValueWrite(startChannel, c255(newValue, 3));
        dmxValueWrite(startChannel + 1, c255(newValue, 2));
        if (isOnOffSpeed(startChannel) || isOnOffSpeed(startChannel + 1)) {
          updSp = onoffSpeed;
        }
        dmxSpeedsWrite(startChannel, updSp);
        dmxSpeedsWrite(startChannel + 1, updSp);
      } else if  (packetBuffer[0 + 3] == 'V') {
        dmxValueWrite(startChannel, c255(newValue, 2));
        dmxValueWrite(startChannel + 1, c255(newValue, 3));
        if (isOnOffSpeed(startChannel) || isOnOffSpeed(startChannel + 1)) {
          updSp = onoffSpeed;
        }
        dmxSpeedsWrite(startChannel, updSp);
        dmxSpeedsWrite(startChannel + 1, updSp);
      } else {
        dmxValueWrite(startChannel, newValue);
        if (isOnOffSpeed(startChannel)) {
          updSp = onoffSpeed;
        }
        dmxSpeedsWrite(startChannel, updSp);
      }
    }

    for (int i = 0; i < UDP_TX_PACKET_MAX_SIZE; i++) {
      packetBuffer[i] = '\0';
    }
  }
}

#ifdef USE_EEPROM
byte readUUIDRegister(byte r) {
  //from http://www.freetronics.com.au/pages/setting-the-mac-address-using-the-mcp-24aa025e48
  unsigned char v;
  Wire.beginTransmission(UUID_I2C_ADDRESS);
  Wire.write(r);  // Register to read
  Wire.endTransmission();

  Wire.requestFrom(UUID_I2C_ADDRESS, 1); // Read a byte
  while (!Wire.available()) {} //wait
  v = Wire.read();
  return v;
}
#endif

int freeRam () {
  extern int __heap_start, *__brkval;
  int v;
  return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval);
}

// ---------------- Arduino LOOP --------------------
void loop() {

  ParseUDP();

  ct = millis();
  dt = timeDiff(lastTime, ct);
  if (dt >= 2) {
    lastTime = ct;
    if (dt > maxdt) {
      maxdt = dt;
    }
    dtcnt = dtcnt + 1;
    dtsum = dtsum + dt;
    stepDMX();
    currStep++;  // bei Ueberlauft 255->0
    if ((currStep % 100) == 0) {
      //  CheckConfigFromSerial();
    }
#ifdef ESP8266
    cnt2 = cnt2 + 1;
    if (cnt2 = 20) {
      //ca. 40ms = 25Hz
      dmx.update();
      cnt2 = 0;
    }
#endif
  }
  cnt = cnt + 1;

  if (cnt >= 1000) {
    DEBUG_PRINTln(F("ALIVE"));
    DEBUG_PRINTln(maxdt);
#ifndef ESP8266
    DEBUG_PRINTln(F("RAM "));
    DEBUG_PRINTln(freeRam());
#endif
    cnt = 0;
    maxdt = 0;
  }

#ifdef USE_DHCP
  Ethernet.maintain();
#endif
}


